#ifndef _LOCOTOOL_H
#define _LOCOTOOL_H
//
// LocoTool - A program to edit Locomotion object description .dat files
//
// Copyright (C) 2004 by Josef Drexler
//
// This file is distributed under the term of the GNU General Public Licence
// version 2 or higher.  Please read the file LICENCE for details.
//
//
// locotool.h - definitions for the locotool.dll exports
//


// define LOCOTOOLCOMPRESS to get access to the RLE codecs
#ifdef LOCOTOOLCOMPRESS
#	include <stdint.h>	// for uint8_t et.al.
#endif

// definition of an error handler
// your function should look like
//
//	void err_handler(char *message);
//
// it's called if there's an error in the processing somewhere
// at that point the processing will be aborted
// (and no cleanup is done; neither of memory nor of any files created)

typedef void LocoTool_err_proc_t(char *);
typedef LocoTool_err_proc_t *LocoTool_err_proc;

// integer and string options
typedef enum {
	opt_palettetype = 0,	// palette type to use when decoding
	opt_demofile = 1,	// whether to generate a demo file checksum
	opt_quiet = 2,		// no output, no message, fail silently
	opt_allinone = 3,	// put all sprites in a single PNG file
	opt_heapdebug = 4,	// to debug the heap functions (write about allocations/deallocations on stderr)
	opt_allunknown = 5,	// print all unknown variables, even empty ones

	opt_INTLAST,
} en_intoption;

typedef enum {
	opt_filebase = 0,	// file path and basename for tmp/xml/dat files
				// will have e.g. ".xml" appended to make the xml filename
	opt_pngbase = 1,	// file path and basename for png files
				// will have e.g. "0043.png" appended to make the png filename

	opt_STRLAST,
} en_stroption;

// get and set integer and string options
//
// get returns the current value
// set sets the new value and returns the old one
//
// LocoTool_set_str_option_copy makes a local copy of the string, call it
// again with str==NULL to free the local copy.  The other calls just keep
// a pointer, so make sure the value stays valid until you're done.
//
// LocoTool_set_str_option_copy does *not* return the old value, because
// that value is free'd just before making a copy of the new value.  Instead
// it returns 1 on success and 0 on error.
//
int LocoTool_get_int_option(en_stroption opt);
int LocoTool_set_int_option(en_stroption opt, int val);

char *LocoTool_get_str_option(en_stroption opt);
char *LocoTool_set_str_option(en_stroption opt, char *str);
int LocoTool_set_str_option_copy(en_stroption opt, char *str);

// decode the given .dat file
// returns 1 if successful, 0 otherwise
int LocoTool_decode(char *datname);

// encode from the give .xml file
// returns 1 if successful, 0 otherwise
int LocoTool_encode(char *xmlname);

// install your own error handler, the default just prints the message on
// stderr and exits
void LocoTool_set_err_handler(LocoTool_err_proc handler);

// get the version string from the DLL
const char *LocoTool_get_version();

// version info as 0xMMmmrrrr (MM=major, mm=minor, rrrr=revision)
int LocoTool_get_vernum();

#ifdef LOCOTOOLCOMPRESS

// access to the raw RLE codec
//
// LocoTool_rledecode RLE decodes data and stores the result and length
// in the outdata and outlen parameters
int LocoTool_rledecode(uint8_t *data, uint32_t len, uint8_t **outdata, uint32_t *outlen);

// LocoTool_rleencode RLE encodes data and stores the result and length
// in the rleout and rlelen parameters
int LocoTool_rleencode(uint8_t *data, uint32_t len, uint8_t **rleout, uint32_t *rlelen);

// the checksum algorithm
uint32_t LocoTool_checksum(uint8_t *data, uint32_t len);

// the chunk compression schemes
//
// string decompression for chunk type 2
int LocoTool_decompress(uint8_t *data, uint32_t len, uint8_t **outdata, uint32_t *outlen);

// string scrambling for chunk type 3
int LocoTool_scramble(uint8_t *data, uint32_t len);

// string descrambling for chunk type 3
int LocoTool_descramble(uint8_t *data, uint32_t len);

#endif

#endif // ndef _LOCOTOOL_H
